/*
  ==============================================================================

    SonicCrypt Seq
    Copyright (C) 2025 Sebastian Sünkler

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU Affero General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU Affero General Public License for more details.

    You should have received a copy of the GNU Affero General Public License
    along with this program.  If not, see <https://www.gnu.org/licenses/>.

  ==============================================================================
*/

#pragma once
#include <JuceHeader.h>
#include "PluginProcessor.h"

// --- HELPER CLASSES ---

class TinyButton : public juce::Button {
public:
    enum Type { Mute, Solo };
    Type type;
    TinyButton(Type t) : juce::Button(t == Mute ? "M" : "S"), type(t) { setClickingTogglesState(true); }
    void paintButton(juce::Graphics& g, bool isMouseOver, bool isButtonDown) override {
        auto b = getLocalBounds().toFloat().reduced(2);
        bool on = getToggleState();
        g.setColour(juce::Colour(0xff1E1E1E).brighter(0.1f));
        if (on) g.setColour(type == Mute ? juce::Colours::red.withAlpha(0.8f) : juce::Colours::yellow.withAlpha(0.8f));
        g.fillRoundedRectangle(b, 3.0f);
        g.setColour(on ? juce::Colours::black : juce::Colours::grey);
        g.setFont(juce::FontOptions(12.0f).withStyle("Bold"));
        g.drawText(type == Mute ? "M" : "S", b, juce::Justification::centred);
        if (isMouseOver) g.drawRoundedRectangle(b, 3.0f, 1.0f);
    }
};

class MidiDragButton : public juce::Button
{
public:
    std::function<void()> onDragStartCallback;
    MidiDragButton() : juce::Button("MidiDrag") {}
    void paintButton(juce::Graphics& g, bool isMouseOver, bool isButtonDown) override {
        g.setColour(juce::Colour(0xff00C4FF).withAlpha(isMouseOver ? 1.0f : 0.7f));
        g.drawRoundedRectangle(getLocalBounds().toFloat().reduced(2), 4.0f, 2.0f);
        g.setFont(juce::FontOptions(14.0f).withStyle("Bold"));
        g.drawText("MIDI", getLocalBounds(), juce::Justification::centred);
        if (isButtonDown) g.fillRoundedRectangle(getLocalBounds().toFloat().reduced(2), 4.0f);
    }
    void mouseDrag(const juce::MouseEvent& e) override {
        if (e.mouseWasDraggedSinceMouseDown() && onDragStartCallback) {
            onDragStartCallback();
        }
    }
};

class AboutOverlay : public juce::Component {
public:
    AboutOverlay() {
        // OK Button
        okButton.setButtonText("OK");
        okButton.setColour(juce::TextButton::buttonColourId, juce::Colours::transparentBlack);
        okButton.onClick = [this] { setVisible(false); };
        addAndMakeVisible(okButton);

        // AGPL Lizenz Link
        licenseLink.setButtonText("GNU AGPL v3 License");
        licenseLink.setURL(juce::URL("https://www.gnu.org/licenses/agpl-3.0.html.en"));
        licenseLink.setColour(juce::HyperlinkButton::textColourId, juce::Colour(0xff00C4FF));
        addAndMakeVisible(licenseLink);

        // JUCE Link ("Made with JUCE")
        juceLink.setButtonText("Made with JUCE");
        juceLink.setURL(juce::URL("https://juce.com"));
        juceLink.setColour(juce::HyperlinkButton::textColourId, juce::Colours::white.withAlpha(0.6f)); // Etwas dezenter
        addAndMakeVisible(juceLink);
    }

    void paint(juce::Graphics& g) override {
        g.fillAll(juce::Colours::black.withAlpha(0.7f));

        // Layout-Berechnung
        auto area = getLocalBounds().toFloat().withSizeKeepingCentre(450, 280); // Etwas höher gemacht für mehr Platz

        // Hintergrund
        g.setColour(juce::Colour(0xff1E1E1E).darker(0.2f));
        g.fillRoundedRectangle(area, 10.0f);
        g.setColour(juce::Colours::black);
        g.drawRoundedRectangle(area, 10.0f, 2.0f);

        auto content = area.reduced(20);

        // Großes Icon links
        auto iconArea = content.removeFromLeft(80).removeFromTop(80);
        iconArea.setY(iconArea.getY() + 20);
        g.setColour(juce::Colour(0xff00C4FF));
        g.fillEllipse(iconArea);
        g.setColour(juce::Colours::black.withAlpha(0.3f));
        g.setFont(juce::FontOptions(50.0f).withStyle("Bold"));
        g.drawText("i", iconArea.toNearestInt(), juce::Justification::centred);

        content.removeFromLeft(20);

        // Überschrift
        g.setColour(juce::Colours::white);
        g.setFont(juce::FontOptions(20.0f).withStyle("Bold"));
        g.drawText("About SonicCrypt Seq", content.removeFromTop(30).toNearestInt(), juce::Justification::topLeft);

        // Entwickler
        g.setFont(juce::FontOptions(15.0f));
        g.drawText(juce::CharPointer_UTF8("Idea & Development: Sebastian S\xc3\xbc\x6ekler"), content.removeFromTop(25).toNearestInt(), juce::Justification::topLeft);

        // Beschreibung
        g.setColour(juce::Colours::grey);
        g.setFont(juce::FontOptions(13.0f));
        g.drawFittedText("A poly-rhythmic multi-layer sequencer.\nVersion 1.0", content.removeFromTop(40).toNearestInt(), juce::Justification::topLeft, 2);

        // Label "Licensed under:"
        g.drawText("Licensed under:", content.removeFromTop(20).toNearestInt(), juce::Justification::topLeft);

        // --- Steinberg Trademark Hinweis (Ganz unten, sehr klein) ---
        // Wir nehmen den unteren Rand des Fensters
        auto footerArea = area.removeFromBottom(20).reduced(10, 0);
        g.setColour(juce::Colours::grey.withAlpha(0.5f));
        g.setFont(10.0f);
        g.drawFittedText("VST is a registered trademark of Steinberg Media Technologies GmbH.", footerArea.toNearestInt(), juce::Justification::centred, 1);
    }

    void resized() override {
        auto area = getLocalBounds().withSizeKeepingCentre(450, 280);

        // OK Button
        okButton.setBounds(area.getX() + 350, area.getBottom() - 45, 80, 25);

        // Lizenz Link (sitzt unter dem Label "Licensed under")
        licenseLink.setBounds(area.getX() + 120, area.getY() + 135, 150, 20);

        // JUCE Link (unten links oder mittig über dem Footer)
        juceLink.setBounds(area.getX() + 20, area.getBottom() - 45, 120, 25);
    }

private:
    juce::TextButton okButton;
    juce::HyperlinkButton licenseLink;
    juce::HyperlinkButton juceLink;
};

class PresetListComponent : public juce::ListBoxModel {
public:
    PresetListComponent(juce::ListBox& lb, std::function<void(juce::File)> onSelect) : listBox(lb), onSelection(onSelect) {}
    int getNumRows() override { return presets.size(); }
    void paintListBoxItem(int rowNumber, juce::Graphics& g, int width, int height, bool rowIsSelected) override {
        if (rowIsSelected) g.fillAll(juce::Colour(0xff00C4FF).withAlpha(0.2f));
        g.setColour(rowIsSelected ? juce::Colour(0xff00C4FF) : juce::Colours::white);
        g.setFont(14.0f); g.drawText(presets[rowNumber].getFileNameWithoutExtension(), 5, 0, width, height, juce::Justification::centredLeft);
    }
    void listBoxItemClicked(int row, const juce::MouseEvent&) override { if (onSelection) onSelection(presets[row]); }
    void refresh(const juce::File& folder) { presets.clear(); if (folder.isDirectory()) for (auto entry : juce::RangedDirectoryIterator(folder, false, "*.xml")) presets.add(entry.getFile()); listBox.updateContent(); }
private: juce::ListBox& listBox; juce::Array<juce::File> presets; std::function<void(juce::File)> onSelection;
};

class LockButton : public juce::Button {
public: LockButton() : juce::Button("Lock") { setClickingTogglesState(true); }
      void paintButton(juce::Graphics& g, bool, bool) override {
          auto bounds = getLocalBounds().toFloat(); bool isLocked = getToggleState();
          g.setColour(isLocked ? juce::Colour(0xff00C4FF) : juce::Colour(0xff888888));
          auto body = bounds.reduced(2.0f); body.removeFromTop(bounds.getHeight() * 0.4f);
          juce::Path p; p.addRoundedRectangle(body, 2.0f);
          float w = body.getWidth() * 0.6f, h = bounds.getHeight() * 0.35f, x = body.getCentreX() - w * 0.5f, y = body.getY() - h * 0.8f;
          if (!isLocked) y -= h * 0.3f; juce::Path s; s.addArc(x, y, w, h, 3.14f, 6.28f, true);
          g.fillPath(p); g.strokePath(s, juce::PathStrokeType(2.0f));
      }
};

// --- MAIN EDITOR ---
class NewProjectAudioProcessorEditor : public juce::AudioProcessorEditor, public juce::Timer, public juce::DragAndDropContainer
{
public:
    NewProjectAudioProcessorEditor(NewProjectAudioProcessor&);
    ~NewProjectAudioProcessorEditor() override;

    void paint(juce::Graphics&) override;
    void resized() override;
    void timerCallback() override;

    void mouseDown(const juce::MouseEvent& e) override;
    void mouseDrag(const juce::MouseEvent& e) override;
    void mouseWheelMove(const juce::MouseEvent& e, const juce::MouseWheelDetails& wheel) override;

private:
    NewProjectAudioProcessor& audioProcessor;
    const int BASE_WIDTH = 1150;
    const int BASE_HEIGHT = 650;

    bool constructorFinished = false;

    juce::Image logoImage;

    const juce::Colour sonicBgDark{ 0xff121212 };
    const juce::Colour sonicHeader{ 0xff1E1E1E };
    const juce::Colour sonicAccent{ 0xff00C4FF };
    const juce::Colour sonicText{ 0xffFFFFFF };

    juce::GroupComponent presetGroup{ "presets", "PRESETS" };
    juce::ListBox presetListBox;
    std::unique_ptr<PresetListComponent> presetListModel;
    juce::TextButton savePresetButton{ "SAVE NEW" };
    juce::TextButton refreshButton{ "REFRESH" };

    juce::ImageButton logoButton{ "link" };
    juce::TextButton copyrightBtn{ "copyright" };
    AboutOverlay aboutOverlay;

    int currentLayerIndex = 0;
    juce::TextButton layerBtnA{ "LAYER A" };
    juce::TextButton layerBtnB{ "LAYER B" };
    juce::TextButton layerBtnC{ "LAYER C" };

    // Mute Solo Buttons
    TinyButton muteBtn{ TinyButton::Mute };
    TinyButton soloBtn{ TinyButton::Solo };

    juce::ToggleButton layerActiveToggle{ "Active" };

    juce::TextButton genAllBtn{ "GEN ALL" };
    juce::TextButton genCurrBtn{ "GEN A" };
    juce::ToggleButton chaosToggle{ "CHAOS MODE" };

    MidiDragButton midiDragBtn;

    juce::ComboBox rateCombo; juce::Label rateLabel{ "rate", "RATE" }; LockButton rateLock;
    juce::ComboBox rootCombo; juce::Label rootLabel{ "root", "ROOT" }; LockButton rootLock;
    juce::Slider transSlider; juce::Label transLabel{ "trans", "TRANS" }; LockButton transLock;
    juce::ComboBox scaleCombo; juce::Label scaleLabel{ "scale", "SCALE" }; LockButton scaleLock;
    juce::Slider stepsSlider; juce::Label stepsLabel{ "steps", "STEPS" }; LockButton stepsLock;
    juce::ComboBox dirCombo; juce::Label dirLabel{ "dir", "MODE" }; LockButton dirLock;
    juce::Slider octSlider; juce::Label octLabel{ "oct", "OCTS" }; LockButton octLock;
    juce::Slider gateSlider; juce::Label gateLabel{ "gate", "GATE" }; LockButton gateLock;
    juce::Slider swingSlider; juce::Label swingLabel{ "swing", "SWING" }; LockButton swingLock;

    std::shared_ptr<juce::FileChooser> fileChooser;

    void randomizePattern(bool allLayers);
    void updateUIFromData();
    void selectLayer(int index);
    void updateFontSize();
    void openSavePresetDialog();

    JUCE_DECLARE_NON_COPYABLE_WITH_LEAK_DETECTOR(NewProjectAudioProcessorEditor)
};